/***************************************************************************
 *
 * Copyright (C) 2001 International Business Machines
 * All rights reserved.
 *
 * This file is part of the GPFS mmfslinux kernel module.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer. 
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution. 
 *  3. The name of the author may not be used to endorse or promote products 
 *     derived from this software without specific prior written
 *     permission. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *************************************************************************** */
/*
 * Interface definitions for basic common services, Linux version
 *
 * Contents:
 *   cxiGetThreadId - get thread identifier
 *   cxiUiomove - transfer data between kernel buffer and user buffer(s)
 *   cxiCopyIn - copy data from user to kernel buffer (kernel service)
 *   cxiCopyOut - copy data from kernel to user buffer (kernel service)
 *   cxiCopyInstr - copy string from user to kernel space (kernel service)
 *
 * $Id: cxiSystem-plat.h,v 1.74.2.2 2002/04/27 00:01:29 mcnabb Exp $
 *
 * $Log: cxiSystem-plat.h,v $
 * Revision 1.74.2.2  2002/04/27 00:01:29  mcnabb
 * Defect 365561: ensure unique id returned by Linux cxiGetThreadID by putting
 * ids returned in user space out of range used for kernel thread ids.
 *
 * Revision 1.74.2.1  2001/11/07 18:24:55  mcnabb
 * Defect 353738: Manage a ref count on the mmfs module for every mount.
 * An additionaly ref will be on the module as long as the daemon is
 * alive.  We're doing this because if the daemon dies with mount
 * points still active, the mmfs module can be unloaded by rmmod but
 * the code doesn't successfully clean everything up.  Thus the rmmod
 * needs to be blocked from the unload attempt by ref counts.
 *
 * Revision 1.74  2001/10/11 14:05:22  gjertsen
 * Provide support for 64-bit mutex lockword for IA64 (LOCKWORD64).
 *
 * Revision 1.73  2001/10/09 00:11:55  eshel
 * add dump_trace_back() routine
 *
 * Revision 1.72  2001/10/03 14:46:17  dcraft
 * First attempt to bring us up to 2.4.9 and 2.4.10
 *
 * Revision 1.71  2001/09/22 20:10:36  dcraft
 * Remove kiobufs from cxiKernelIODescriptor_t.  Use temporary
 * kiobufs for map/unmap.   Remove dead code and dead comments
 * in portability layer and update readmes and license.
 * Fix traceback to appear in mmfs.log file.
 *
 * Revision 1.70  2001/09/20 21:47:01  wyllie
 * Add call to determine if a cxiBlockingMutex_t has any threads waiting
 * for it
 *
 * Revision 1.69  2001/09/20 17:55:07  gjertsen
 * Fix up ifdefs so things break at compile time for new platforms.
 *
 * Revision 1.68  2001/09/12 23:54:13  eshel
 * Reanme cxi_schedule() to cxiYield()
 *
 * Revision 1.67  2001/09/12 22:21:13  eshel
 * add cxi_schedule() to call schedule()
 *
 * Revision 1.66  2001/09/06 14:01:03  dixonbp
 * cxi routines to get/set lock owner.
 *
 * Revision 1.65  2001/09/04 19:36:26  gjertsen
 * Use 2M stack size for IA64 (pthreads default).
 *
 * Revision 1.64  2001/08/22 01:14:13  jpalmer
 * Correct value of FL_OFFSET_MAX to longlong for fcntl lock length 0 test
 *
 * Revision 1.63  2001/08/21 14:48:21  mroberts
 *
 * Rename cxiIsSambaThread to cxiIsSambaOrLockdThread.
 * Create cxiIsSambaThread that tests only for smbd and not lockd.
 *
 * Revision 1.62  2001/08/10 00:41:44  wyllie
 * Use lock indices instead of lock numbers when calling cxiBlockingMutexInit,
 * to simplify statistics gathering.  Add call to reset or read statistics
 * for cxiBlockingButex_ts from the kernel.
 *
 * Revision 1.61  2001/08/09 21:11:45  dcraft
 * Modifications to allow running on latest Redhat 7.1 update
 * Kernel version 2.4.3-12.
 * Requires checkout of new site.mcr.proto
 *
 * Revision 1.60  2001/08/06 23:39:09  wyllie
 * Add routine to determine if the current process holds the kernel lock.
 *
 * Revision 1.59  2001/07/30 15:45:18  dixonbp
 * Add missing update to the linux prototype for cxiCanUncacheOSNode.
 *
 * Revision 1.58  2001/07/21 23:33:23  eshel
 * Make some code common for the daemon and kernel or add new kernel version of
 * daemon code so we can do fast open in the kernel when ever the file is found
 * in the stat cache.
 *
 * Revision 1.57  2001/07/19 23:25:31  dcraft
 * Modified linux trace to allow non blocking trace record
 * writes (format is TRACE?N).  New gpfs swapd process created
 * which is responsible for reclaiming inodes (5 percent every
 * time it runs).  Marked all our inodes so that they would be
 * ignored by linux kswapd.  Added "unused" inode to inode
 * cache that could be used as a signal that linux kswapd is
 * running and kick off gpfs swapd.  Added means to ignore attempts
 * to kill mmfsd by kswapd if the system gets low on memory.
 * All done in an attempt to avoid kswapd premature wakeup on simple
 * locks and mutexes.
 *
 * Revision 1.56  2001/07/13 19:56:43  wyllie
 * Get PAGE_OFFSET by asking the mmfslinux_... kernel module rather than
 * compiling the constant into proprietary code.  Rename constants needed to
 * determine the GPFS memory map and export them to non-proprietary files.
 *
 * Revision 1.55  2001/07/02 18:01:10  eshel
 * Initialize gpfsNode hash table based on (fileCacheLimit + statCacheLimit).
 *
 * Revision 1.54  2001/06/06 21:56:22  wyllie
 * Change the way NFSData objects are timed out.  Instead of using watchdog
 * timers that enqueue NFSData objects at interrupt level for later
 * processing by the NFS watchdog kernel process, just let the watchdog
 * process wake up periodically and examine its LRU list of NFSData objects
 * to decide which should be closed.  This fixes a bug on Linux that shows
 * up as the assert "nfsP->nextWatchP == NFS_WATCH_NOT_QUEUED" under high
 * load.  This also allows deleting a bunch of cxiXXX interfaces dealing
 * with watchdog timers.
 *
 * Revision 1.53  2001/05/11 20:13:09  eshel
 * Cleanup some GPFS_LINUX ifdefs.
 *
 * Revision 1.52  2001/05/02 00:21:32  schmuck
 * Fix another problem found by tortureDir test on Linux:
 * On lookup and create, instantiate the dcache entry while holding the
 * inode lock, or, in case of a negative dcache entry, the directory lock.
 * This closes a window where a token revoke could clear the
 * CO_VFS_REFERENCE flag without invalidating the dcache entry.
 * It also eliminates the need for a d_revalidate operation.
 *
 * Revision 1.51  2001/04/08 22:18:53  dcraft
 * Fix multinde delete race conditions.  Still incomplete.
 *
 * Revision 1.50  2001/04/06 20:59:39  gjertsen
 * Selectively screen out regular sync calls from kupdate.
 *
 * Revision 1.49  2001/03/07 20:13:09  jpalmer
 * SMB Open Lock function
 *
 * Revision 1.48  2001/03/05 23:28:25  dcraft
 * Modify inode and gpfsNode reference management.  Inode is now acquired
 * during gpfsNode creation and must be released via cxiPutOSNode().
 * (documented in gpfs database).  Add "mmfsadm dump vnodes" for producing
 * trace info on all held inodes.
 *
 * Revision 1.47  2001/02/28 17:27:11  wyllie
 * Move definition of EXTERNC to cxiTypes.h
 *
 * Revision 1.46  2001/02/15 21:57:35  manoj
 * Support for NSS (NAS Security System). The filesystem stores NT-style ACLs and
 * uses NSS for access control and permission checking. All NSS code should be
 * #ifdef'ed by CONFIG_NSS.
 *
 * Revision 1.45  2001/01/25 18:34:05  wyllie
 * Rename panic to cxiPanic, which is implemented as an exported entry point of
 * the mmfslinux module on Linux, and as a macro that invokes panic on AIX.
 *
 * Revision 1.44  2001/01/08 21:22:27  wyllie
 * Use explicit LINUX_KERNEL_VERSION instead of LATEST_LINUX_KERNEL to allow
 * better control over code that depends of the level of the Linux kernel.
 *
 * Revision 1.43  2001/01/08 20:49:04  gjertsen
 * Minor code cleanup. No functional changes.
 *
 * Revision 1.42  2000/12/15 13:57:20  gjertsen
 * Clean up documentation.
 *
 * Revision 1.41  2000/12/14 19:26:53  mcnabb
 * Typo in last checkin
 *
 * Revision 1.40  2000/12/14 18:13:08  mcnabb
 * Fix for defect 324281: UNABLE TO VIEW A GPFS FILE ON SOME NODES
 * If a compacted inode is looked up, the generation value being looked for
 * was not being compared with the generation in the inode. This resulted
 * in the gnode type field being incorrect, and the read system call
 * changed the call to readdir since the gnode type indicated that the
 * inode was a directory.
 * Add verification of generation numbers for compacted inodes.
 * Inode status and mode extraction routines from compacted inode.
 *
 * Revision 1.39  2000/12/13 18:35:52  wyllie
 * Move definition of cxiBlockingMutex_t type to cxiTypes-plat.h.
 *
 * Revision 1.38  2000/12/13 00:07:01  eshel
 * Update GPFS_LINUX_SEM_SIZE for ia64.
 *
 * Revision 1.37  2000/12/12 20:23:17  gjertsen
 * Make picky IA64 compiler happy in 1002 gnu toolchain.
 *
 * Revision 1.36  2000/12/12 16:36:12  wyllie
 * Change e_sleep_thread and cxiEventWakeup... to cxiWaitEventWait and
 * cxiWaitEventSignal/Broadcast.  Rename cxiWaitEventActive to
 * cxiWaitEventHasWaiters.
 *
 * Revision 1.35  2000/12/08 18:34:33  wyllie
 * Add cxiBlockingMutex_t type and operations that act on it.  Uses simple_lock
 * on AIX and binary semaphores on Linux.
 *
 * Revision 1.34  2000/12/06 16:52:09  dcraft
 * write inline versions of cxiString functions for linux kernel
 *
 * Revision 1.33  2000/11/09 19:11:31  dcraft
 * No need for NSD_BLKGETSIZE
 *
 * Revision 1.32  2000/11/09 15:19:52  gjertsen
 * Definition for PAGE_SIZE is now pulled in here.
 *
 * Revision 1.31  2000/11/08 15:28:23  gjertsen
 * Fix up NSD header include.
 *
 * Revision 1.30  2000/11/02 19:46:59  gjertsen
 * Linux code split.
 *
 * Revision 1.29  2000/10/24  14:06:54  gjertsen
 * Clean up linux module specific code so that gpfs can be
 * compiled in the kernel (to allow IA64 kernel debugging).
 *
 * Revision 1.28  2000/10/19  19:09:58  dixonbp
 * NFSWatchKprocInit for linux.
 *
 * Revision 1.27  2000/10/02  20:42:49  wyllie
 * Clean up multi-platform mutex code.  Add assembler versions of the fast path
 * for mutex acquire and release for i386.
 *
 * Revision 1.26  2000/09/15  22:56:53  dcraft
 * Defines for where vi editor and file system file can be found
 *
 * Revision 1.25  2000/09/07  21:35:03  dcraft
 * Make tsfattr/tsattr work on linux.  Make kxPutACL/kxGetACL
 * routines and ACL commands work on linux.  Split kx routines
 * into OS dependent directories.
 *
 * Revision 1.24  2000/08/29  18:33:16  dcraft
 * Produce mmfs module.
 *
 * Revision 1.23  2000/08/28  14:15:26  gjertsen
 * Use C wrapper functions for C++ class methods in
 * cxi interface. Convert all cxi functions to C interface.
 *
 * Revision 1.22  2000/08/23  15:13:22  dixonbp
 * Define some cxi macros needed to support NFS on linux.
 *
 * Revision 1.21  2000/08/21  22:16:13  dcraft
 * Create cxiDev_t type that is based on user level dev_t.  Provide
 * mapping functions between kernel, user, and inode device field.
 * Fix NLS yes/no query.
 *
 * Revision 1.20  2000/08/08  16:57:04  gjertsen
 * Need prototype for gettimeofday
 *
 * Revision 1.19  2000/08/07  22:33:45  dcraft
 * Use new cxiGetTOD() time of day function for second,
 * nanosecond needs.  Define 32 bit time types and start
 * using in place of current on disk time_t types (necessary
 * for 64 bit).  Use HiResTime where possible.
 *
 * Revision 1.18  2000/07/27  15:59:25  dcraft
 * Fix thread signal masks (actually process signal masks since
 * each linux thread is actually a cloned process).  Signals
 * generated by the terminal now work correctly.   Turn on
 * NLS catalog message retrieval.
 *
 * Revision 1.17  2000/07/25  16:18:59  gjertsen
 * Pull some function prototypes out of kernel only define area (locking stuff).
 *
 * Revision 1.16  2000/07/24  14:13:23  gjertsen
 * Introduce cxiCopyin, cxiCopyout, and cxiCopyInstr as general kernel ops.
 *
 * Revision 1.15  2000/07/20  15:49:34  dcraft
 * Abstract event words for OS independence
 *
 * Revision 1.14  2000/07/11  16:36:08  wyllie
 * Use cxiUio_t instead of struct uio.  Use cxiUiomove instead of uiomove.  Use
 * CXI_READ instead of UIO_READ, etc.
 *
 * Revision 1.13  2000/06/29  04:56:52  mcnabb
 * AIX setpri priority numbers 0-99 (as recorded in the mmfs.cfg file)
 * will be decremented by 60 (PUSER+20) to get to the setpriority range
 * of -20 to 20. Linux setpriority will then bound anything outside that
 * range to the edges of the range.
 *
 * Revision 1.12  2000/06/28  12:47:43  gjertsen
 * Now use eflock_t in platform independant code due to problem with
 * compiling struct eflock when eflock is already typedef'ed to a struct.
 * Now use cxiSetPri and cxiGetPri instead of native AIX calls setpri and getpri.
 *
 * Revision 1.11  2000/06/23  20:31:40  dcraft
 * Remove sysdep.h and ModeBits.h.  Pare down some OS specific
 * header includes.  Use cxiMode_t instead of mode_t due to
 * linux short mode_t in kernel.
 *
 * Revision 1.10  2000/06/20  23:53:51  dcraft
 * Remove OS specifics from mmfsnode.C and mmfsnode.h. Remove sharkvfs.h.
 * Remove unnecessary OS specific ifdefs of header files.
 *
 * Revision 1.9  2000/06/13  14:50:21  dixonbp
 * Remove cxiFcntlLock prototype (moved to cxiSystem.h).
 *
 * Revision 1.8  2000/06/12  21:42:41  dixonbp
 * Move cxiFlock_t from cxiTypes-plat to cxiTypes.
 *
 * Revision 1.7  2000/06/12  19:32:42  dcraft
 * Rename SharkPrivateVFSData to gpfsVFSData_t and split off into OS independent
 * vfsdata.h.   OS dependent layer relies only on forward declaration gpfsVFSData_t
 * since it uses it only as a pointer (hence the non cxi name).
 *
 * Revision 1.6  2000/06/08  21:31:25  dcraft
 * Make unmount OS independent.  Correct OS independent mount problems.
 *
 * Revision 1.5  2000/06/08  13:06:36  gjertsen
 * Introduce cxi types used at user-level and kernel extension
 * interfaces for xmem and uio. Purged out gpfs_flock with cxiFlock_t.
 *
 * Revision 1.4  2000/06/06  20:04:43  dixonbp
 * Prototype for cxiFcntlLock.
 *
 * Revision 1.3  2000/06/05  19:01:32  dcraft
 * Complete mount callbacks within the OS dependent kernel extension.
 * Record vfs number in gpfsVfsData_t.
 *
 * Revision 1.2  2000/05/30  21:26:05  wyllie
 * Move types from cxiSystemTypes.h to cxiTypes.h.  Use cxi prefix instead of kxi.
 *
 * Revision 1.1  2000/05/19  23:19:48  wyllie
 * Linux version of routine to get current thread ID.
 *
 */

#ifndef _h_cxiSystem_plat
#define _h_cxiSystem_plat


/* Size of stack to allocate for GPFS threads */
#ifdef GPFS_ARCH_I386
#define GPFS_STACK_SIZE (256*1024)
#endif /* GPFS_ARCH_I386 */
#ifdef GPFS_ARCH_IA64
#define GPFS_STACK_SIZE (2*1024*1024)
#endif /* GPFS_ARCH_IA64 */

/* -------------------------------------------------------------- */
/* Interfaces usable from both kernel code and process-level code */
/* -------------------------------------------------------------- */

/* Main module device name */
#define GPFS_DEVNAME "/dev/ss0"

/* Virtual memory page size.  Used for aligning I/O buffers. */
/* NOTE: Shark-gpl.h grabs PAGE_SIZE for the portability layer */
#if !defined(GPFS_GPL)
/* Note that the page size can be adjusted in the IA64 Linux kernel */
#include <sys/user.h>     /* PAGE_SIZE */
#endif /* !GPFS_GPL */

/* Process operations */
#if !defined(GPFS_GPL)
EXTERNC pid_t getpid() __THROW;
#else
/* use of THROW causes conflicts in portability kernel code */
EXTERNC pid_t getpid();
#endif /* !GPFS_GPL */

/* Get the kernel thread ID, which is guaranteed to be non-zero.  In a
 * process, this just calls getpid.  Within the kernel, this calls a routine
 * that extracts the process ID from the current process block.
 */
#define MAX_GPFS_KERNEL_TID 32767      /* max kernel thread id (2^15 - 1) */
#ifdef _KERNEL
EXTERNC cxiThreadId cxiGetThreadId();

/* Attempts to kill process due to low paging space, generate
 * SIGTERM signal.
 */
EXTERNC int cxiLowPagingNotify(pid_t pid);
#else   /* not _KERNEL */
#include <unistd.h>
#define cxiGetTid() ((cxiThreadId)getpid())
/* Ensure that user cxiGetThreadId generates an id that is disjoint from
 * the kernel thread ids (current->pid) which range up to MAX_GPFS_KERNEL_TID.
 * A mask is applied to the pthread id to remove it from this range.
 */
#define GPFS_PTID_MASK 0x80000000               /* pthread id/handle mask */
#define cxiGetThreadId() ((cxiThreadId)(pthread_self() | GPFS_PTID_MASK))
#endif  /* _KERNEL */


#ifdef _KERNEL
/* Convert a kernel stack address to the thread ID of the thread that uses
   that stack */
EXTERNC int cxiStackAddrToThreadId(char* stackP, cxiThreadId* tidP);
/* Convert a kernel thread pointer to the corresponding thread ID */
EXTERNC int cxiThreadPtrToThreadId(char* threadP, cxiThreadId* tidP);
#endif  /* _KERNEL */

#ifdef _KERNEL
/* Transfer data from buffer(s) in user space to or from a buffer in the
 *  kernel.
 */
EXTERNC int cxiUiomove(char* kBufP, unsigned long nBytes, Boolean toKernel,
                       struct cxiUio_t* uioP);

EXTERNC void * cxiGetFcntlOwner(eflock_t *flP);
#define cxiSetEflockOwner(eflockP, flP) (eflockP->l_owner = cxiGetFcntlOwner(flP))
#define FL_OFFSET_MAX (loff_t)(~0ULL>>1)
#endif  /* _KERNEL */


#define cxiDevMajor(D)       ((Int32)(((UInt32)(D) >> 8) & 0xFF))
#define cxiDevMinor(D)       ((Int32)((D) & 0xFF))

/* Extra defines not on other platforms to massage it back down
 * to a linux kernel specific 16 bits.
 */
#define cxiKernelDevToDev(D) ((cxiDev_t)(D))
#define cxiDevToKernelDev(D) \
  ((cxiKernelDev_t)((cxiDevMajor(D) << 8) | (cxiDevMinor(D) & 0xFF)))

/* Translate to GPFS native disk format which used high and low 16 bits
 * for major/minor
 */
#define cxiDevToDev32(D) \
  (cxiDev32_t)((cxiDevMajor(D) << 16) | (cxiDevMinor(D) & 0xFFFF))

#define cxiDev32ToDev(D) \
  (cxiDev_t)((cxiDev32Major(D) << 8) | (cxiDev32Minor(D) & 0xFF))

/* System priority operations */

/* Define priority constants inherited from AIX;  priority can only be
 * relatively adjusted in Linux and not set directly.
 * AIX setpri priority numbers 0-99 (as recorded in the mmfs.cfg file)
 * will be decremented by 60 (PUSER+20) to get to the setpriority range
 * of -20 to 20. Linux setpriority will then bound anything outside that
 * range to the edges of the range.
 */
#define PRI_SCHED    16
#define PUSER        40

#define cxiSetPri(pid, newpri) (setpriority(PRIO_PGRP, (int)pid, \
                                            (int)newpri-(PUSER+20)))
#define cxiGetPri(pid) ((PUSER+20) + getpriority(PRIO_PGRP, (int)pid))

/* Threads are actually a cloned process with separate signal masks */
#define cxiSigThreadMask(a,b,c)  sigprocmask((a),(b),(c));

/* Assertion functions */
#include <assert.h>
#define cxiAssertFailed(STRING,FILE,LINE) \
          __assert_fail((STRING),(FILE),(LINE),__ASSERT_FUNCTION)

void dump_trace_back(char *failingExpr, char *srcFileName, int srcLineNumber);

#ifdef INSTRUMENT_LOCKS
/* Statistics kept per lock class: number of lock acquisitions and the
   number of times the lock appeared to be held before it was requested */
struct BlockingMutexStats
{
  UInt32 bmsAcquires;                   /* number of times lock obtained */
  UInt32 bmsConflicts;                  /* number of lock conflicts */
};

#define MAX_GPFS_LOCK_NAMES 48

#ifdef _KERNEL
EXTERNC struct BlockingMutexStats BlockingMutexStatsTable[MAX_GPFS_LOCK_NAMES];
#endif
#endif  /* INSTRUMENT_LOCKS */



#ifdef _KERNEL
/* String functions for the kernel */
static inline size_t cxiStrlen(const char *str1)
{
  const char *beg1;

  if (str1 == NULL)
    return 0;

  for (beg1 = str1; *str1; str1++);

  return str1 - beg1;
}

static inline char *cxiStrcpy(char *str1, const char *str2)
{
  char *beg1 = str1;

  while (*str1++ = *str2++);

  return beg1;
}

static inline char *cxiStrcat(char *str1, const char *str2)
{
  char *beg1;

  for (beg1 = str1; *str1; str1++);

  while (*str1++ = *str2++);

  return beg1;
}

static inline int cxiStrcmp(const char *str1, const char *str2)
{
  for (; *str1 == *str2; str1++, str2++)
    if (*str1 == '\0')
      return 0;

  return *str1 - *str2;
}

static inline char *cxiStrncpy(char *str1, const char *str2, size_t num)
{
  char *beg1 = str1;

  while (num)
  {
    num--;
    if ((*str1++ = *str2++) == '\0')
      break;
  }

  /* strncpy is defined to null out the total length */
  while (num)
  {
    num--;
    *str1++ = '\0';
  }

  return beg1;
}

static inline char *cxiStrncat(char *str1, const char *str2, size_t num)
{
  char *beg1;

  for (beg1 = str1; *str1; str1++);

  while (num-- && (*str1 = *str2++))
   str1++;

  *str1 = '\0';

  return beg1;
}

static inline char *cxiStrrchr(const char *str1, int ch)
{
  char *ret = NULL;

  do
  {
    if (*str1 == (char)ch)
      ret = (char *)str1;
  }
  while (*str1++);

  return ret;
}

static inline int cxiMemcmp(const void *v1, const void *v2, size_t num)
{
  char *str1 = (char *)v1;
  char *str2 = (char *)v2;

  for (; num && (*str1 == *str2); str1++, str2++, num--);

  if (num)
    return *str1 - *str2;
  else
    return 0;
}

static inline void *cxiMemcpy(void *v1, const void *v2, size_t num)
{
  char *str1 = (char *)v1;
  char *str2 = (char *)v2;

  while (num--)
    *str1++ = *str2++;

  return v1;
}

static inline void *cxiMemset(void *v1, int c, size_t num)
{
  char *str1 = (char *)v1;

  while (num--)
    *str1++ = c;

  return v1;
}
#endif  /* _KERNEL */

/* Local file system list */
#define FILESYSTEM_FILE "/etc/fstab"
#define VI_FILE         "/bin/vi"
#define ROOT_GROUP      "root"
#define ROOT_USER       "root"

/* AIX inherited constants from device.h */
#define CFG_INIT 0x1
#define CFG_TERM 0x2

#ifdef _KERNEL
/* AIX inherited constants from sleep.h */
/* flags for e_thread_sleep */
#define INTERRUPTIBLE           (1)     /* sleep is interruptible */
#define LOCK_HANDLER            (2)     /* lock used by interrupt handlers */
//#define LOCK_READ             (4)     /* complex lock read mode specified */
#define LOCK_SIMPLE             (8)     /* simple lock specified */
//#define LOCK_WRITE            (16)    /* complex lock write mode specified */
/* return values from e_thread_block */
#define THREAD_AWAKENED         (1)     /* normal wakeup value */
#define THREAD_TIMED_OUT        (2)     /* sleep timed out */
#define THREAD_INTERRUPTED      (4)     /* sleep interrupted by signal */
#define THREAD_OTHER            (4096)  /* Beginning of subsystem values */
#endif /* _KERNEL */


/* --------------------------------------- */
/* Interfaces usable only from kernel code */
/* --------------------------------------- */

#ifdef _KERNEL

/* Macros to convert Linux kernel version to a decimal integer that can
   be compared to the symbol LINUX_KERNEL_VERSION.
   See the description of LINUX_KERNEL_VERSION in site.mcr.proto. */
#define GPFS_KERNEL_VERSION(_major,_minor,_sub,_mod) \
   ( (_major)*1000000 + (_minor)*10000 + (_sub)*100 + (_mod))

/*
 * Kernel memory allocation services:
 *
 * These are implemented as subroutines in gpl-linux/kxiSystem.C.  All
 * allocations use storage class GFP_KERNEL.  There is no distinction
 * between pinned and unpinned kernel memory in Linux, so the unpinned
 * interfaces are defined in terms of the pinned interfaces as macros.
 */
EXTERNC void* cxiMallocPinned(int nBytes);
#define cxiMallocUnpinned(nBytes)  cxiMallocPinned(nBytes)
EXTERNC void cxiFreePinned(void* p);
#define cxiFreeUnpinned(p)         cxiFreePinned(p)

EXTERNC void *cxiBigMalloc(int size);
EXTERNC void cxiBigFree(char *ptr);


/* cxiIsSuperUser - return true if caller has maximum authorization (is root) */
EXTERNC Boolean cxiIsSuperUser();

/* cxiGetMaxFileSize - return the maximum file size the calling process
 * is allowed to create.  A value of cxiUnlimitedFileSize means file
 * sizes are not limited.  On Linux, there is no limit on file sizes.
 */
#define cxiUnlimitedFileSize MAX_INT64
#define cxiGetMaxFileSize() (cxiUnlimitedFileSize)

/* NFS services for Linux */
/* nfsd calls open and release for every file read/write.  Provide a way
   to detect that nfsd is the caller and avoid the extra overhead. */
EXTERNC Boolean cxiIsNFSThread();

EXTERNC Boolean cxiIsGPFSThread();

/* kupdate services for Linux */
/* Use to identify syncs from kupdate that are generated way
   too often by Linux (every 5 seconds). */
EXTERNC Boolean cxiIsKupdateThread();

#ifdef SMB_LOCKS
/* Samba services for Linux */
/* smbd calls open and subsequently makes calls to set open deny modes and
   op locks.  Provide a way to distinguish these Samba calls from calls
   from other users for which default actions on open and op locks must
   be taken */
EXTERNC Boolean cxiIsSambaOrLockdThread();
#endif

#ifdef CCL
/* Samba services for Linux */
/* Provide a way to distinguish Samba calls from calls
   from other users */
EXTERNC Boolean cxiIsSambaThread();
#endif

/* The nfsd threads are allowed to FSYNC their writes. */
#define cxiAllowNFSFsync() cxiIsNFSThread()

/* OS dependent VFS calls */
#define cxiDisconnectVfs(PVP)   /* nothing ??? */

EXTERNC void cxiDeleteMmap(cxiVmid_t);  /* delete memory map */

/* Set OS dependent virtual node type and device */
EXTERNC void cxiSetOSNodeType(struct cxiNode_t *cnP, cxiMode_t mode,
                              cxiDev_t rdev);
#define cxiOSNodeTypeIs(CNP,VTYPE) (((CNP)->nType) == (VTYPE))

EXTERNC Boolean cxiCanUncacheOSNode(void *osVfsP, struct cxiNode_t *cnP, void **vPP);

EXTERNC void cxiAddOSNode(void *dentryP, void *vP);

/* Kernel operations to copy data between user and kernel space */
EXTERNC int cxiCopyIn(char *from, char *to, unsigned long size);
EXTERNC int cxiCopyOut(char *from, char *to, unsigned long size);
EXTERNC int cxiCopyInstr(char *from, char *to, unsigned long size,
                         unsigned long *len);

/* Adhere to XPG 1170 behaviour */
#define cxiIsXPG()	1

/* Routines to manipulate cxiWaitEvent_t objects.  The e_sleep_thread and
   cxiWaitEventWakeup... routines are deprecated.  Use cxiWaitEventWait and
   cxiWaitEventSignal/Broadcast instead. */
EXTERNC void cxiWaitEventInit(cxiWaitEvent_t* weP);
EXTERNC Boolean cxiWaitEventHasWaiters(cxiWaitEvent_t* weP);
EXTERNC void cxiWaitEventWakeup(cxiWaitEvent_t* weP);
EXTERNC void cxiWaitEventWakeupOne(cxiWaitEvent_t* weP);
EXTERNC void cxiWaitEventWakeupResult(cxiWaitEvent_t* weP, int rc);
EXTERNC int e_sleep_thread(cxiWaitEvent_t *weP, Simple_lock *sl, int waitFlags);

/* Routines to manipulate Simple_locks.  These are deprecated; use
   cxiBlockingMutex_t instead of Simple_lock where possible. */
EXTERNC void lock_alloc(Simple_lock *, int, int, int);
EXTERNC void simple_lock_init(Simple_lock *);
EXTERNC void lock_free(Simple_lock *);
EXTERNC void simple_lock(Simple_lock *);
EXTERNC void simple_unlock(Simple_lock *);
EXTERNC void unlock_enable(int, Simple_lock *);
EXTERNC int disable_lock(int, Simple_lock *);


/* Kernel-only synchronization primitives.  A cxiBlockingMutex_t can be used
   to protect a critical section.  Requests to acquire a cxiBlockingMutex
   that is already held by another thread will block the caller.  Must
   initialize the mutex before acquiring it, and must terminate it before
   deallocating the storage it occupies.  Implemented using semaphores on
   Linux. */
EXTERNC void cxiBlockingMutexInit(cxiBlockingMutex_t* mP, int bmNameIdx);
EXTERNC void cxiBlockingMutexAcquire(cxiBlockingMutex_t* mP) REGPARMS;
EXTERNC void cxiBlockingMutexRelease(cxiBlockingMutex_t* mP) REGPARMS;
EXTERNC void cxiBlockingMutexTerm(cxiBlockingMutex_t* mP);
EXTERNC Boolean cxiBlockingMutexHeldByCaller(cxiBlockingMutex_t* mP);
EXTERNC Boolean cxiBlockingMutexHasWaiters(cxiBlockingMutex_t* mP);

/* Yield the CPU to allow other processes to run */
EXTERNC void cxiYield();

/* Well defined portability interface calling changing (depending on kernel
 * level) linux interface
 */
EXTERNC int cxiFillDir(void *argP, const char *nameP, 
                       int namelen, offset_t offset, ino_t ino);

/* A cxiWaitEvent_t can be used like a condition variable using the calls
   below.  There must be an associated cxiBlockingMutex_t held when calling
   cxiWaitEventWait that will be reacquired before returning.  The routine
   cxiWaitEventSignal wakes up one thread waiting in cxiWaitEventWait, while
   cxiWaitEventBroadcast wakes all such waiting threads.  The flags parameter
   to cxiWaitEventWait indicates whether or not the wait can be interrupted
   by a signal.  It should be set to INTERRUPTIBLE if interrupts due to
   signals are desired, otherwise set to 0. */
EXTERNC int cxiWaitEventWait(cxiWaitEvent_t* weP, cxiBlockingMutex_t* mutexP,
                             int waitFlags);
EXTERNC void cxiWaitEventSignal(cxiWaitEvent_t* weP);
EXTERNC void cxiWaitEventBroadcast(cxiWaitEvent_t* weP);
EXTERNC void cxiWaitEventBroadcastRC(cxiWaitEvent_t* weP, int rc);

/* Kill the system due to a fatal error */
EXTERNC void cxiPanic(const char* panicStrP);

/* Get the address of the first byte not addressible by processes */
EXTERNC UIntPtr cxiGetKernelBoundary();

/* Return true if this process holds the big kernel lock (BKL) */
EXTERNC Boolean cxiHoldsBKL();

EXTERNC void cxiExportModuleStruct(void *modAddr);
EXTERNC void cxiIncModuleCounter(int up);

#ifdef CONFIG_NSS
EXTERNC gid_t * cxiGetGroupList(int *numgroups);
EXTERNC int cxiGetFSUid();
EXTERNC int cxiGetFSGid();
EXTERNC gid_t * cxiGetGroupListPID(pid_t pid, int *numgrps);
EXTERNC int cxiSetGroupListPID(pid_t pid, gid_t *groups, int numgrps);
EXTERNC int cxiSetPAG(pid_t pid, gid_t g1, gid_t g2);
EXTERNC Boolean cxiIsValidPID(pid_t pid);
EXTERNC int cxiIsCapable(int type);
#endif

#endif  /* _KERNEL */

/* Return current time of day in seconds and nanoseconds. */
#ifdef _KERNEL
EXTERNC int cxiGetTOD(cxiTimeStruc_t *tP);

static inline void cxiInvalidateAttr(cxiNode_t *cnP, int what)
{
  cnP->icValid &= ~what;
  if (what & CXI_IC_PERM)
    cxiInvalidatePerm(cnP);
}

#else /* !_KERNEL */

EXTERNC int gettimeofday __P ((struct timeval *__tv, struct timezone *__tz));
/* cxiTimeStruc_t should have same size as linux struct timeval.
 * Call LGPL library function and translate microseconds from call
 * to nanoseconds.
 */
EXTERNC inline int cxiGetTOD(cxiTimeStruc_t *tP)
{
  int rc = gettimeofday((struct timeval *)tP, NULL);
  tP->tv_nsec = tP->tv_nsec * 1000;
  return rc;
}

/* String functions in user space */
#define cxiStrcpy  strcpy
#define cxiStrncpy strncpy
#define cxiStrcat  strcat
#define cxiStrncat strncat
#define cxiStrrchr strrchr
#define cxiStrcmp  strcmp
#define cxiMemcpy  memcpy
#define cxiStrlen  strlen
#define cxiMemcmp  memcmp

#endif /* _KERNEL */

/* Misc module redefines for IA64 debugging */
#if defined(_KERNEL) && !defined(MODULE)
#define MOD_INC_USE_COUNT
#define MOD_DEC_USE_COUNT
#define MOD_IN_USE 0
#endif /* _KERNEL && !MODULE */

#endif  /* _h_cxiSystem_plat */

